package com.rameses.enterprise.treasury.cashreceipt;

import com.rameses.rcp.annotations.*;
import com.rameses.rcp.common.*;
import com.rameses.osiris2.client.*;
import com.rameses.osiris2.common.*;

class MultiCashReceiptModel extends BasicCashReceipt {

    @Invoker 
    def invoker;
    
    @Service('MultiCashReceiptService')
    def multiReceiptSvc;
    
    String entityName = "misc_cashreceipt_multi"
    
    void init() {
        // change the collection type handler if set in Invoker properties 
        def shandler = invoker?.properties.handler; 
        if ( shandler ) entity.collectiontype.handler = shandler; 
        
        super.init(); 
    } 
    
    def post() {
        if( entity.amount <= 0 ) 
            throw new Exception("Please select at least an item to pay");
        if( entity.totalcash + entity.totalnoncash == 0 )
            throw new Exception("Please make a payment either cash or check");
            
        def numformat = new java.text.DecimalFormat('0.00'); 
        entity.totalcash = new BigDecimal( numformat.format( entity.totalcash )); 
        entity.cashchange = new BigDecimal( numformat.format( entity.cashchange )); 
        entity.totalnoncash = new BigDecimal( numformat.format( entity.totalnoncash )); 
        entity.amount = new BigDecimal( numformat.format( entity.amount )); 
        if( (entity.totalcash-entity.cashchange) + entity.totalnoncash != entity.amount )
            throw new Exception("Total cash and total non cash must equal the amount");    
            
        if(entity.balancedue > 0)
            throw new Exception("Please ensure that there is no balance unpaid");

        validateBeforePost();
        
        int itemsperpage = -1; 
        try {
            itemsperpage = Integer.parseInt( entity.collectiontype.itemsperpage.toString()); 
        } catch(Throwable t){} 
        
        if ( itemsperpage <= 0 ) {
            itemsperpage = 1000; 
        }
        
        entity.itemsperpage = itemsperpage;
        return postReceipts(); 
    } 
    
    void postReceipts() {
        int itemsperpage = entity.itemsperpage;
        boolean multi_pages = ( entity.items.size() <= itemsperpage ? false : true );

        def receipts = null; 
        if ( multi_pages ) {
            receipts = buildReceipts( itemsperpage ); 
        }
        else {
            receipts = [ entity ];
        }
        
//        def amount = entity.amount;
//        def totalcash = entity.totalcash;
//        def totalnoncash = entity.totalnoncash;
//        if ( totalcash > 0 && totalnoncash > 0 ) { 
//            throw new Exception("Please select only 1 method of payment"); 
//        }
        
        def series = entity.series; 
        def endseries = entity.endseries; 
        def qtybalance = (endseries - series) + 1; 
        if ((qtybalance - receipts.size()) < 0) {
            def msg = "This transaction requires "+ receipts.size() + 
                      " receipts but the remaining qty balance of your AF is "+ qtybalance; 
            throw new Exception( msg ); 
        }

        if ( multi_pages ) { 
            applyFinalAdjustments( receipts ); 
        } 

        def receiptnos = receipts.collect{ it.receiptno }.join(", "); 

        boolean pass = false;
        def h = { pass = true; }
        Modal.show("cashreceipt_confirm", [ handler:h, receiptno: receiptnos ]);
        if ( !pass ) return null; 
        
        boolean postok = false;
        try { 
            beforePost();
            entity._paymentorderid = _paymentorderid; 
            def res = multiReceiptSvc.post( entity, receipts ); 
            
            entity.receipts = (res ? res : []); 
            
            postok = true; 
        } 
        catch(e) { 
            e.printStackTrace(); 
            postError(); 
            throw e; 
        }

        if ( postok ) {
            completed = true; 
            binding.fireNavigation('completed'); 
            binding.refresh(); 
        } 
        
        try {
            def fo = ( entity.receipts ? entity.receipts.first() : null ); 
            entity._options = (fo?._options == null ? [:] : fo._options);

            if(entity.txnmode.equalsIgnoreCase("ONLINE") && mainProcessHandler==null) { 
                print();
            } 
        }
        catch(e) {
            e.printStackTrace();
            MsgBox.alert("warning! no form handler found for.  " + entity.formno +". Printout is not handled" );
        }

        if( mainProcessHandler ) { 
            mainProcessHandler.forward( entity );
        }
        return null;         
    }
    
    void print() {
        // printing of receipts is handled by the mainProcessHandler
        // when forward method is called 
    } 
    
    final def buildReceipts( int itemsperpage ) {
        def receipts = [];
        entity.items.eachWithIndex{ o,idx-> 
            int idxno = idx+1;
            int idxpg = idxno / itemsperpage; 
            if ((idxno % itemsperpage) > 0) idxpg++; 

            def rct = null; 
            if ( receipts.size() < idxpg ) {
                rct = [:]; 
                entity.each{ k,v-> 
                    if ( v instanceof Collection ) {
                        //do not copy 
                    } else if ( v instanceof Map ) {
                        //do not copy 
                    } else {
                        rct.put(k, v); 
                    }
                }
                rct.collectiontype = entity.collectiontype; 
                rct.collector = entity.collector;
                rct.payer = entity.payer;
                rct.user = entity.user; 
                rct.org = entity.org; 
                rct.paymentitems = [];
                rct.shares = [];
                rct.items = [];
                receipts << rct; 
            } 
            else { 
                rct = receipts[idxpg-1];
            } 

            rct.items << o; 
        } 
        return receipts; 
    }

    final void applyFinalAdjustments( receipts ) {
        def series = entity.series; 
        def endseries = entity.endseries; 
        def qtybalance = (endseries - series) + 1; 
        if ((qtybalance - receipts.size()) < 0) {
            def msg = "This transaction requires "+ receipts.size() + 
                      " receipts but the remaining qty balance of your AF is "+ qtybalance; 
            throw new Exception( msg ); 
        }

        def total_amount = entity.amount; 
        def total_noncash = entity.totalnoncash; 
        def total_cash = Math.max( 0.0, total_amount - total_noncash );
        
        def eft = null; 
        def checks = []; 
        if ( total_noncash > 0 ) {
            def pitems = entity.paymentitems?.findAll{( it.reftype == 'CHECK' && it.check?.objid )}
            if ( pitems ) {
                pitems.collect{ it.check }.groupBy{ it.objid }.each{ k,v-> 
                    v.each{ 
                        it.split = 1; 
                        it._balance = it.amount;                         
                    }
                    checks << v.first(); 
                } 
                checks.each{ chk-> 
                    def tmp = pitems.find{( it.refid == chk.objid )} 
                    chk._template = [ 
                        refid: tmp.refid, reftype: tmp.reftype, 
                        refno: tmp.refno, refdate: tmp.refdate, 
                        particulars: tmp.particulars 
                    ]; 
                } 
            } 

            pitems = entity.paymentitems?.findAll{( it.reftype == 'EFT' && it.item?.objid )} 
            if ( pitems ) { 
                def tmp = pitems.first(); 
                eft = tmp.item; 
                eft._template = [ 
                    refid: tmp.refid, reftype: tmp.reftype, 
                    refno: tmp.refno, refdate: tmp.refdate, 
                    particulars: tmp.particulars 
                ]; 
            } 
        } 

        def serieslength = entity.serieslength; 
        def prefix = (entity.prefix ? entity.prefix : ''); 
        def suffix = (entity.suffix ? entity.suffix : ''); 

        receipts.each{ rct-> 
            rct.series = series; 

            def skey = rct.controlid +'-'+ rct.series.toString(); 
            rct.objid = 'RCT-'+ com.rameses.util.Encoder.MD5.encode( skey ); 
            rct.receiptno = prefix + rct.series.toString().padLeft(serieslength,'0') + suffix; 
            rct.amount = rct.items.sum{ it.amount } 
            rct.totalnoncash = 0.0; 
            rct.totalcash = 0.0; 
            rct.cashchange = 0.0; 
            rct.paymentitems = []; 
            rct.checks = []; 
            rct.eft = null; 

            if ( total_noncash > 0 ) { 
                rct.totalcash = 0.0; 
                rct.totalnoncash = 0.0;

                def groupItems = rct.items.groupBy{[ objid: it.item.fund.objid ]}
                if ( checks ) { 
                    groupItems.each{ k,v-> 
                        def fund = v.first().item.fund; 
                        k.putAll( fund ); 
                    }
                    applyCheckPayment( rct, groupItems, checks ); 
                } 
                else if ( eft ) { 
                    groupItems.each{ k,v-> 
                        def pmt = [item: [:]]; 
                        pmt.putAll( eft._template ); 
                        pmt.item.putAll( eft ); 
                        pmt.item.remove('_template'); 
                        pmt.amount = v.sum{ it.amount } 
                        pmt.fund = k; 
                        rct.paymentitems << pmt; 
                        rct.totalnoncash += pmt.amount; 
                    } 
                } 
                rct.totalcash = rct.amount - rct.totalnoncash; 
                groupItems.clear();  
            }
            else {
                rct.totalnoncash = 0.0; 
                rct.totalcash = rct.amount; 
            }
            series++; 
        } 
        
        // verify and validate the payment items 
        receipts.each{ rct-> 
            if ( rct.totalnoncash > 0 ) {
                def pmt_amt = rct.paymentitems.sum{( it.amount ? it.amount : 0.0 )} 
                if ( pmt_amt == null ) pmt_amt = 0.0; 

                if ( rct.totalnoncash != pmt_amt ) {
                    throw new Exception("Payment items total amount must be equal to the total non-cash");
                }
                
                if ( rct.amount != ( rct.totalnoncash + rct.totalcash )) {
                    throw new Exception("Receipt amount must be equal to the sum of non-cash and cash");
                }
            }
        }
    }
    
    void applyCheckPayment( rct, groupItems, checks ) {
        def decformat = new java.text.DecimalFormat("0.00"); 
        def chk = checks.find{( it._balance > 0 )} 
        def chkamt = chk?._balance; 
        if ( !chkamt ) return; 

        groupItems.each{ k,v-> 
            if ( chk ) {
                if ( !rct.checks.find{( it.objid == chk.objid )}) { 
                    rct.checks << chk; 
                } 

                def amt = v.sum{ it.amount } 
                rct.paymentitems.each{
                    if ( it.fund.objid == k.objid ) {
                        amt -= it.amount; 
                    }
                }
                
                def rate = amt / rct.amount;
                rate = new BigDecimal( decformat.format( rate )).doubleValue();

                def pmt = [check: [:]]; 
                pmt.putAll( chk._template ); 
                pmt.check.putAll( chk ); 
                pmt.check.remove('_template'); 
                pmt.check.remove('_balance'); 
                
                def rctbal = rct.amount - rct.totalnoncash; 
                if ( chkamt > rctbal ) {
                    pmt.amount = amt; 
                } else {
                    pmt.amount = ( chkamt * rate ); 
                }
                pmt.fund = k; 
                rct.paymentitems << pmt; 
                rct.totalnoncash += pmt.amount; 
                chk._balance -= pmt.amount; 
            }
        } 

        if ((rct.amount - rct.totalnoncash) > 0 ) {
            if ( checks.find{( it._balance > 0 )} ) {
                applyCheckPayment( rct, groupItems, checks )
            }
        }        
    }
}
