package com.rameses.gov.etracs.rptis.report.models;

import com.rameses.rcp.annotations.*
import com.rameses.rcp.common.*
import com.rameses.osiris2.client.*
import com.rameses.osiris2.reports.*;
import com.rameses.gov.etracs.rptis.util.*;
import com.rameses.etracs.shared.*;

public abstract class AbstractCertificationModel
{
    @Service('ReportParameterService')
    def paramSvc;

    @Service('RPTCertificationService')
    def certificationSvc;

    @Service('UserService')
    def userSvc;
    
    @Service('Var')
    def var

    @Service('DateService')
    def dtSvc;

    @Binding 
    def binding;
    
    @Invoker
    def inv;
    
    def opener;

    def MODE_CREATE = 'create';
    def MODE_SELECT = 'select';
    def MODE_READ   = 'read';
    
    def entity;
    def mode;
    def showincludemembers  = false;
    def reportPath = 'com/rameses/gov/etracs/rptis/reports/'
    
    def certificationTypes;
    boolean showasofyear = false;
    boolean isonline = false;
    def areatypes = [
        [code: 'sqm', title: 'square meter'],
        [code: 'ha', title: 'hectare'],
    ]

    
    abstract def getService();
    abstract String getReportName();
    
        
    def getReportData(){
        saveSignatures()
        return entity;
    }
    
    SubReport[] getSubReports() {
        return null;
    }
    
    Map getParameters(){
        return [:];
    }

    
    def getTitle(){
        return 'Certification of ' + inv.caption 
    }
    
    def createEntity(){
        def map = [:];
        def types = certificationTypes;
        map.certtype = (types ? types[0] : null);
        map.properties = [:];
        return map;
    }
    
    void afterInit(){}
    
    def init() {
         entity = createEntity();
        entity.objid            = RPTUtil.generateId('RC');
        entity.opener           = (inv.properties.opener ? inv.properties.opener : opener);
        entity.online           = isonline;
        entity.certifiedby      = var.get("ASSESSORCERTIFIEDBY");
        entity.certifiedbytitle = var.get("ASSESSORCERTIFIEDBYTITLE");
        entity.byauthority      = var.get("ASSESSORBYAUTHORITY");
        entity.byauthoritytitle = var.get("ASSESSORBYAUTHORITYTITLE");
        entity.attestedby      = var.get("ASSESSORATTESTEDBY");
        entity.attestedbytitle = var.get("ASSESSORATTESTEDBYTITLE");
        entity.purpose          = var.get("TDTRUECOPYPURPOSE");
        if (!entity.purpose){
            entity.purpose = "whatever legal purposes it may serve him/her"; 
        }
            
        entity.includemembers   = false;
        entity.asofyear         = dtSvc.getServerYear();
        entity.oramount         = 0.0;
        entity.stampamount      = 0.0;
        entity.office           = 'assessor';
        entity.official         = false;
        officialuse             = false;
        afterInit();
        mode = MODE_CREATE;
        binding?.refresh('entity.*');
        return 'default'
    }
    
    def getCertificationTypes(){
        return [
            [type:'byfaas', caption:'By FAAS'],
            [type:'bytaxpayer', caption:'By Taxpayer'],
        ]
    }
    
    def open(){
        entity = service.openCertification(entity.objid);
        opener = entity.opener;
        mode = MODE_READ;
        return doPreview();
    }

    void afterLookupTaxpayer() {
        
    }
    
    def getLookupTaxpayer(){
        return InvokerUtil.lookupOpener('entity:lookup',[
            onselect : { 
                entity.taxpayer = [objid:it.objid, name:it.name, address:it.address.text];
                entity.requestedby = it.name;
                entity.requestedbyaddress = it.address.text;
                afterLookupTaxpayer();
                binding.refresh('entity.taxpayer.*');
                binding.requestFocus('entity.taxpayer');
            },
            onempty  : { 
                entity.taxpayer = null;
                entity.requestedby = null;
                entity.requestedbyaddress = null;
            },
        ])
    }
    

    void validateFaas(faas) {
        if (faas.state != 'CURRENT' && faas.state != 'CANCELLED'){
            throw new Exception('FAAS is not current or cancelled.')
        }
    }
    
    def getLookupFaas(){
        return InvokerUtil.lookupOpener('faas:lookup',[
            onselect : { 
                validateFaas(it);
                entity.faasid = it.objid;
                entity.faas = it;
                entity.tdno= it.tdno;
                entity.trackingno = it.trackingno;
                entity.taxpayer = it.taxpayer;
                entity.requestedby = it.taxpayer.name;
                entity.requestedbyaddress = it.taxpayer.address;
            },
            onempty  : { 
                entity.faasid = null;
                entity.tdno= null;
                entity.taxpayer = null;
                entity.requestedby = null;
                entity.requestedbyaddress = null;
            },
        ])
    }
       
    def officialuse = false;

    void setOfficialuse( officialuse ){
        this.officialuse = officialuse;
        entity.official = officialuse;
        entity.orno = ( officialuse ) ? 'Official Use' : '';
        entity.ordate = ( officialuse ) ? null : entity.serverDate;
        entity.oramount = ( officialuse ) ? 0.0 : entity.oramount;
        entity.stampamount = ( officialuse ) ? 0.0 : entity.stampamount;
    }


    def report =[
        getReportName : { return getReportName() },
        getSubReports : { return getSubReports() },
        getReportData : { return getReportData() },
        getParameters : { 
            def params = paramSvc.getStandardParameter() + getParameters(); 
            params.GIVENAT = params.LGUADDRESS
            params.LOGOLGU = EtracsReportUtil.getInputStream("lgu-logo.png")
            params.LOGOBLGF = EtracsReportUtil.getInputStream("lgu-blgf.png")
            params.LOGOASSESSOR = EtracsReportUtil.getInputStream("lgu-assessor.png")            
            return params 
        },
        isAllowPrint: {
            if (!entity.online) return true;
            if (entity.task?.state == 'released' && 
                certificationSvc.checkUserAllowedAction([action: 'rptcertification.reprint'])) {
                    return true;
                }
            return 'releaser'.equalsIgnoreCase(entity.task?.state)
        },
        isAllowSave: {
            if (!entity.online) return true;
            return 'releaser'.equalsIgnoreCase(entity.task?.state)
        }
    ] as ReportModel;

    void beforeSave(){}
    
    def save(){
        beforeSave();
        return service.createCertification( entity )
    }
    
    public def doSave(){
        def retval = null
        if (MsgBox.confirm('Save certification?')) {
            entity.putAll(save());
            if (isonline) {
                entity = service.openCertification(entity.objid);
                def op = Inv.lookupOpener("rptcertification:open", [entity:entity] );
                op.target = "topwindow";
                return op;
            } else {
                mode = MODE_READ;
                retval = doPreview();
            }
        }
        return retval;
    }
    
    
    def doPreview(){
        report.viewReport();
        return "preview";
    }

    void saveSignatures() {
        def sigdata = certificationSvc.getSignatories([objid: entity.objid]);
        if (sigdata) {
            entity.putAll(sigdata);
        }
        entity.signatories.each{ k, v ->
            def objid = v.objid + '-' + v.state 
            if (v.signature?.image)
                v.signatureis = DBImageUtil.getInstance().saveImageToFile(objid, v.signature.image)
        }
    }

}