package com.rameses.gov.etracs.bpls.models;

import com.rameses.rcp.annotations.*
import com.rameses.rcp.common.*
import com.rameses.osiris2.client.*
import com.rameses.osiris2.common.*
import com.rameses.util.*;
import java.rmi.server.*;
        
/****************************************************
 * capture types:
 *   info - basic capture info only for current year
 *   unpaid - business has remaining balance
 *   processing - application still unfinished
 *   delinquent - has delinquent
 *   datasource - use datasource
 *****************************************************/
class BusinessDataMigrationModel extends PageFlowController {
        
    @Service("DateService")
    def dateSvc;
            
    @Service("BusinessCaptureService")
    def service;
            
    @Script("BusinessApplicationUtil")
    def application;
            
    @Script("BusinessVerificationUtil")
    def verifyList;
            
    @Script("BusinessOwnerUtil")
    def owner;
            
    @Script("BusinessLobUtil")
    def lob;
        
    @Script("BusinessAddressUtil")
    def address;

    @Script("BusinessCaptureUtil")
    def capture;
            
    @Script("BusinessInfoUtil")
    def appinfo;
                
    @Script("BusinessAssessmentInfoUtil")
    def assessmentInfo;
            
    @FormTitle
    def title = "Business Data Migration";
            
    def entity;
    def receivables;
            
    @Service("BusinessUserRoleService")
    def userRoleSvc;
            
    def start() {
        capture.loadSources();
        assessmentInfo.show_paid = true;
        return super.start();
    }
            
    void init() {
        capture.validateTxnType(); 
        application.init([ txnmode:'CAPTURE']);
                                        
        switch( capture.type ) {
        case 'current':
            entity.apptype = capture.current_apptype;
            entity.appyear = dateSvc.getServerYear();
            entity.state = 'ACTIVE';
            break;
        case 'processing':     
            entity.apptype = capture.processing_apptype;
            entity.dtfiled = capture.processing_dateapplied;
            entity.appyear = dateSvc.getServerYear();
            entity.state = 'PROCESSING';
            break;
        case 'delinquent_renew':     
            entity.apptype = 'RENEW';
            entity.appyear = capture.delinquent_renew_year;
            entity.activeyear = entity.appyear;
            entity.yearstarted = null;
            entity.state = 'ACTIVE';
            if(entity.appyear < 2000 )
            throw new Exception("Last year renewed must be greater than 2000");
            if( entity.appyear >= dateSvc.getServerYear() )
            throw new Exception("Please choose a year before the current year");
            break;
        case 'delinquent_new':     
            entity.apptype = 'NEW';
            entity.appyear = capture.delinquent_new_year;
            entity.activeyear = entity.appyear;
            entity.yearstarted = null;
            if(entity.appyear < 2000 )
            throw new Exception("Last year renewed must be greater than 2000");
            entity.state = 'ACTIVE';
            if( entity.appyear > dateSvc.getServerYear() )
            throw new Exception("Please choose a year before the current year");
            break;
        }    
        lob.reset();
        lob.lobAssessmentType = entity.apptype;
    }

    void openEntry() {
        capture.openEntry();
        lob.reset();
        lob.lobAssessmentType = entity.apptype;
        def r = service.getReceivables(entity);
        entity.taxfees = r.receivables;
        entity.totals =  [regfee:0,othercharge:0,total:0,tax:0];
        entity.txnmode = 'CAPTURE';
        taxfeeModel.reload();
    }
            
    void initAddressType() {
        if( application.copyOwnerAddress ) {
            owner.reload();
            entity.business.address = entity.business.owner.address;
            address.addressType = entity.business.address.type;
        }
        else {
            address.addressType = application.addressType;
        }
    }
            
    def taxfeeModel = [
        fetchList: { o->
            return entity.taxfees;
        },
        onColumnUpdate: { o,col-> 
            o.total = o.amount - (o.amtpaid + o.surcharge + o.interest) - o.discount;
            binding.refresh();
        }
    ] as EditorListModel;
                        
    def paymentModel = [
        fetchList: { o->
            return entity.payments;
        }
    ] as BasicListModel;
            
                        
    void resetAll() {
        if( MsgBox.confirm("This will clear changes and payments made. Continue?")) {
            initAssessment();
        }
    }
            
    void initAssessment() {
        if( capture.type == 'datasource' ) {
            if( !entity.taxfees ) MsgBox.alert('No receivables found');
            return;
        }    
        entity.payments = [];
        entity.taxfees = [];
        taxfeeModel.reload();
        paymentModel.reload();
        binding.refresh();
    }
            
    void initReceivable() { 
        entity.taxfees.each{tf-> 
            def o = entity.lobs.find{ it.oldlobid==tf.oldlob?.objid } 
            if (o) tf.lob = [objid: o.lobid, name: o.name] 
        } 
    } 
            
    def getTotalPayment() {
        return entity.payments.sum{it.total};
    }
            
    void save() {
        if ( !entity.taxfees ) { 
            if (!MsgBox.confirm("You are about to save this record without building the ledger. Proceed?")) { 
                throw new BreakException(); 
            } 
        } else if( !MsgBox.confirm("You are about to save this record. Proceed?") ) {
            throw new BreakException();
        }
                
        entity.taxfees.each{
            if ( !it.account?.objid ) { 
                throw new Exception('Please specify the account mapping for ' + it.oldaccount.title);  
            } 
        } 
        application.save();
    }
            
    void addAnother() {
        entity = [:];
    }
            
    def mapAccount() {
        def params = [
            onselect: {o-> 
                assessmentInfo.selectedTaxfee.account = o;
                taxfeeModel.refresh(); 
            }, 
            onempty: {
                assessmentInfo.selectedTaxfee.account = null;
                taxfeeModel.refresh(); 
            } 
        ]; 
        return mapAccount( params );
    }
            
    def mapAccount( params ) {
        if(!assessmentInfo.selectedTaxfee)
        throw new Exception("Select an item to map");
                    
        return Inv.lookupOpener( "revenueitem:lookup", params );
    } 
            
    def getLookupAccount() {
        def params = [
            onselect: {o-> 
                assessmentInfo.selectedTaxfee.account = o;
            }, 
            onempty: {
                assessmentInfo.selectedTaxfee.account = null;
            } 
        ]; 
        return mapAccount( params );
    } 
}
