package com.rameses.treasury.common.models;

import com.rameses.rcp.common.*;
import com.rameses.rcp.annotations.*;
import com.rameses.osiris2.client.*;
import com.rameses.osiris2.common.*;
import com.rameses.rcp.framework.ClientContext;
import com.rameses.seti2.models.CrudFormModel;
import com.rameses.util.BreakException;

class PaymentOrderModel extends CrudFormModel {

    @Service( dynamic=true )
    def dynaSvc; 
    
    def paymentOrderTypes;
    def paymentOrderType;
    def pageMode = "default";

    public String getConnection() {
        return caller.connection;
    }

    String create() {
        paymentOrderTypes = []; 
        
        def m = [_schemaname: "paymentorder_type"];
        m.where = ["state='ACTIVE' AND accesstype=0"];
        m.orderBy = "title";
        
        def types = queryService.getList( m ); 
        def ctx = ClientContext.getCurrentContext();
        def sec = (ctx == null ? null : ctx.getSecurityProvider()); 
        if ( sec ) {
            types.each{
                def domain = (it.domain ? it.domain.toString().trim() : '');
                def role = (it.role ? it.role.toString().trim() : '');

                if ( !domain ) domain = null; 
                if ( !role ) role = null; 
                
                boolean with_security = ( domain || role ); 
                if ( with_security ) {
                    try { 
                        if ( !role ) role = "*";
                        if ( sec.checkPermission( domain.toString(), role, null )) {
                            paymentOrderTypes << it; 
                        }
                    } 
                    catch(Throwable t) {;}
                }
                else {
                    paymentOrderTypes << it; 
                }
            }
        }
        
        if ( !paymentOrderTypes ) {
            MsgBox.alert("There are no available Payment Order Types or you don't have enough permissions to perform this action");
            return "_close"; 
        }

        super.create();
        entity.items = [];
        entity.state = 'DRAFT'; 
        pageMode = "initial";
        return "initial";
    }

    def doNext() {
        entity.type = paymentOrderType;
        if ( !entity.type )
            throw new Exception("Select the type of PaymentOrder first");

        loadPOType(); 
        pageMode = "default";
        return "default";
    }
    
    public void afterOpen() {
        loadPOType(); 
    }
    
    def queryFilter;
    void loadPOType() {
        def ct = entity.type.collectiontype;
        if ( !ct?.objid )
            throw new Exception("No CollectionType set for this type of PaymentOrder");

        def m = [ _schemaname: "paymentorder_type", objid: entity.type.objid ];
        def pot = persistenceSvc.read( m ); 
        ct = pot.collectiontype; 
        
        queryFilter = [:];
        queryFilter.put( "collectiontype" , ct );
        if ( ct.fund?.objid ) {
            queryFilter.put( "fundid", ct.fund.objid );
        }
    }
    
    
    //this is required for the item component. display here the collectiontypeid used by the lookup
    def getQuery() {
        return queryFilter;
    }
    
    def eventHandler = [
        isColumnEditable: { o,name-> 
            return mode.toString().matches('create|edit'); 
        },
        onColumnUpdate: { o,name-> 
            
        }
    ]; 
    
    @PropertyChangeListener
    def listener = [
        "entity.payer": { o->
            entity.paidby = o.name;
            entity.paidbyaddress = o.address.text;
            binding.refresh();
        }
    ];
    
    
    def save() { 
        if ( !entity.items ) {
            MsgBox.err("Please add at least 1 item in the list"); 
            return null; 
        }
        if ( !entity.amount || entity.amount <= 0 ) {
            MsgBox.err("Total amount must be greater than zero"); 
            return null;             
        }

        return super.save(); 
    } 
    
    def items_before_save = null; 

    public void beforeSave( def mode ) {
        items_before_save = null; 
        if ( mode == 'create' ) {
            items_before_save = entity.items; 
        }
    }
    
    public void afterSave() {
        if ( items_before_save != null ) {
            entity.items = items_before_save; 
        }
    }


    def preview() {
        def state = entity.state.toString().toUpperCase(); 
        if ( state == 'DRAFT' ) {
            MsgBox.alert("No available print form for DRAFT record"); 
        }
        else if ( state == 'CANCELLED' ) {
            MsgBox.alert("No available print form for CANCELLED record"); 
        }
        else if ( state.toString().matches('OPEN|PAID|CLOSED')) {
            return preview('paymentorder_form_report');
        }
        else {
            MsgBox.alert("No available print form for CANCELLED record"); 
        }

        return null; 
    }
    
    def getPrintFormData( Map options ) { 
        return entity; 
    } 
    
    def getReportForm() { 
        def path = 'reports/paymentorder'; 
        return [
            mainreport: path + '/bill.jasper', 
            subreports: [
                [name: 'Items', template: path +'/billitem.jasper']
            ]
        ];
    } 
    
    
    void approve() {
        if ( entity.state.toString() != 'DRAFT' ) {
            MsgBox.err("Record status must be in DRAFT mode"); 
            return;
        }
        
        boolean pass = MsgBox.confirm("You are about to approve this record. Continue?");
        if ( !pass ) return; 
        
        def param = [ _schemaname: getSchemaName(), _action: 'approve' ];
        param.objid = entity.objid;
        param.state = 'OPEN';
        persistenceSvc.update( param ); 
        entity.state = param.state; 

        if ( callbackListHandler ) {
            callbackListHandler.doSearch();
        }
    }
    
    void sendEmail() {
        boolean pass = false;
        def param = [ email: entity.email ]; 
        param.handler = { arg->
            param.to = arg.email;
            pass = true;
        }
        Modal.show("sys_specify_email", param);
        if ( !pass ) return;

        def email = (param.to ? param.to : entity.email); 
        
        if ( !entity.email && param.to ) {
            // update email
            def svc = getPersistenceService();
            def ureq = [ _schemaname: getSchemaName() ]; 
            ureq.objid = entity.objid;
            ureq.email = param.to; 
            svc.update( ureq );

            entity.email = ureq.email;
        }

        def svc = dynaSvc.lookup('OnlinePaymentOrderService'); 
        def map = [ objid: entity.objid, type: 'for-payment', email: email ];
        svc.sendMail( map );
        MsgBox.alert("Message sent");
    }
    
    def markAsCancelled() {
        if ( entity.state.toString() != 'OPEN' ) {
            MsgBox.err("Record status must be in OPEN mode"); 
            return null; 
        }
        
        boolean pass = MsgBox.confirm("You are about to cancel this record. Continue?");
        if ( !pass ) return null; 
        
        def param = [ _schemaname: getSchemaName(), _action: 'cancel' ];
        param.objid = entity.objid;
        param.state = 'CANCELLED';
        persistenceSvc.update( param ); 
        entity.state = param.state; 

        try {
            if ( callbackListHandler ) {
                callbackListHandler.doSearch();
            }
        } 
        finally {
            return '_close'; 
        }
    }

    def markAsClosed() {
        if ( entity.state.toString() != 'PAID' ) {
            MsgBox.err("Record status must be in PAID mode"); 
            return null; 
        }
        
        boolean pass = MsgBox.confirm("You are about to mark this record as CLOSED. Continue?");
        if ( !pass ) return null; 
        
        def param = [ _schemaname: getSchemaName(), _action: 'close' ];
        param.objid = entity.objid;
        param.state = 'CLOSED';
        persistenceSvc.update( param ); 
        entity.state = param.state; 

        try {
            if ( callbackListHandler ) {
                callbackListHandler.doSearch();
            }
        } 
        finally {
            return '_close'; 
        }
    }

    def delete() {
        if ( !entity.state.toString().matches('CLOSED|CANCELLED')) {
            MsgBox.err("Record status must be in CLOSED or CANCELLED mode"); 
            return null; 
        }
        
        boolean pass = MsgBox.confirm("You are about to delete this record. Continue?");
        if ( !pass ) return null; 
        
        def param = [ _schemaname: getSchemaName(), _action: 'delete' ];
        param.objid = entity.objid;
        param.state = 'DELETED';
        persistenceSvc.update( param ); 
        entity.state = param.state; 

        try {
            if ( callbackListHandler ) {
                callbackListHandler.doSearch();
            }
        } 
        finally {
            return '_close'; 
        }
    }
} 